<?php
/**
 * @license GNU AGPL version 3 or any later version
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace OCA\SideMenu\Controller;

use OC;
use OC\App\AppStore\Fetcher\CategoryFetcher;
use OC\URLGenerator;
use OCA\SideMenu\Service\AppRepository;
use OCA\SideMenu\Service\ConfigProxy;
use OCP\AppFramework\Controller;
use OCP\AppFramework\Http\JSONResponse;
use OCP\IL10N;
use OCP\IRequest;
use OCP\IUserSession;
use OCP\L10N\IFactory;

class NavController extends Controller
{
    /**
     * @var ConfigProxy
     */
    protected $config;

    /**
     * @var AppRepository
     */
    protected $appRepository;

    /**
     * @var IFactory
     */
    protected $l10nFactory;

    /**
     * @var CategoryFetcher
     */
    protected $categoryFetcher;

    /**
     * @var URLGenerator
     */
    protected $router;

    public function __construct(
        string $appName,
        IRequest $request,
        ConfigProxy $config,
        AppRepository $appRepository,
        CategoryFetcher $categoryFetcher,
        URLGenerator $router,
        IL10N $trans,
        IFactory $l10nFactory)
    {
        parent::__construct($appName, $request);

        $this->config = $config;
        $this->appRepository = $appRepository;
        $this->categoryFetcher = $categoryFetcher;
        $this->l10nFactory = $l10nFactory;
        $this->router = $router;
    }

    /**
     * @NoAdminRequired
     * @NoCSRFRequired
     * @PublicPage
     *
     * @return JSONResponse
     */
    public function items()
    {
        $apps = $this->appRepository->getVisibleApps();
        $currentLanguage = substr($this->l10nFactory->findLanguage(), 0, 2);
        $categoriesLabels = $this->categoryFetcher->get();
        $externalSitesInTopMenu = $this->config->getAppValueBool('external-sites-in-top-menu', 0);
        $user = OC::$server[IUserSession::class]->getUser();
        $appsCategories = [];
        $categoriesAppsCount = [];
        $items = [];

        if (!$user) {
            return new JSONResponse([
                'items' => $items,
            ]);
        }

        $topMenuApps = $this->config->getAppValueArray('top-menu-apps', '[]');
        $userTopMenuApps = $this->config->getUserValueArray($user, 'top-menu-apps', '[]');

        if (!empty($userTopMenuApps)) {
            $topMenuApps = $userTopMenuApps;
        }

        foreach ($categoriesLabels as $k => $category) {
            $categoriesLabels[$category['id']] = $category['translations'][$currentLanguage]['name'] ?? $category['translations']['en']['name'];

            unset($categoriesLabels[$k]);
        }

        $categoriesLabels['external_links'] = $this->l10nFactory->get('external')->t('External sites');

        $items['other'] = [
            'name' => '',
            'apps' => [],
        ];

        foreach ($apps as $app) {
            if (in_array($app['id'], $topMenuApps)) {
                continue;
            }

            $categories = (array) $app['category'];
            $appsCategories[$app['id']] = [];

            if (empty($categories)) {
                $categories = ['other'];
            }

            foreach ($categories as $category) {
                if (!isset($items[$category])) {
                    $items[$category] = [
                        'name' => $categoriesLabels[$category],
                        'apps' => [],
                    ];
                }

                if (!isset($categoriesAppsCount[$category])) {
                    $categoriesAppsCount[$category] = 0;
                }

                ++$categoriesAppsCount[$category];

                $appsCategories[$app['id']][] = $category;

                $items[$category]['apps'][$app['id']] = [
                    'name' => $app['name'],
                    'href' => $app['href'],
                    'icon' => $app['icon'],
                ];
            }
        }

        arsort($categoriesAppsCount);

        $keys = array_keys($categoriesAppsCount);

        foreach ($appsCategories as $app => $appCategories) {
            $smallerIndex = count($categoriesAppsCount) - 1;

            foreach ($appCategories as $appCategory) {
                $appKey = array_keys($keys, $appCategory)[0];

                if ($appKey < $smallerIndex) {
                    $smallerIndex = $appKey;
                }
            }

            $category = $keys[$smallerIndex];

            foreach ($items as $itemCategory => $value) {
                if ($itemCategory !== $category && isset($value['apps'][$app])) {
                    unset($items[$itemCategory]['apps'][$app]);

                    if (empty($items[$itemCategory]['apps'])) {
                        unset($items[$itemCategory]);
                    }
                }
            }
        }

        foreach ($items as $category => $value) {
            if (empty($items[$category]['apps'])) {
                unset($items[$category]);
            } else {
                uasort($items[$category]['apps'], function ($a, $b) {
                    return ($a['name'] < $b['name']) ? -1 : 1;
                });
            }
        }

        usort($items, function ($a, $b) {
            return ($a['name'] < $b['name']) ? -1 : 1;
        });

        return new JSONResponse([
            'items' => $items,
        ]);
    }
}
