<?php
/**
 * @license GNU AGPL version 3 or any later version
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace OCA\SideMenu\Controller;

use OC;
use OC\User\User;
use OCA\SideMenu\AppInfo\Application;
use OCA\SideMenu\Service\ConfigProxy;
use OCP\AppFramework\Controller;
use OCP\AppFramework\Http\Response;
use OCP\AppFramework\Http\TemplateResponse;
use OCP\IRequest;
use OCP\IUserSession;

class CssController extends Controller
{
    /**
     * @var ConfigProxy
     */
    protected $config;

    /**
     * @var User
     */
    protected $user;

    public function __construct(string $appName, IRequest $request, ConfigProxy $config)
    {
        parent::__construct($appName, $request);

        $this->user = OC::$server[IUserSession::class]->getUser();
        $this->config = $config;
    }

    /**
     * @NoAdminRequired
     * @NoCSRFRequired
     * @PublicPage
     *
     * @return Response
     */
    public function stylesheet()
    {
        $response = new TemplateResponse(Application::APP_ID, 'css/stylesheet', $this->getConfig(), 'blank');
        $response->addHeader('Content-Type', 'text/css');

        return $response;
    }

    protected function getConfig(): array
    {
        $backgroundColor = $this->config->getAppValue('background-color', '#333333');
        $backgroundColorTo = $this->config->getAppValue('background-color-to', $backgroundColor);

        $topMenuApps = $this->config->getAppValueArray('top-menu-apps', '[]');

        if ($this->user) {
            $userTopMenuApps = $this->config->getUserValueArray($this->user, 'top-menu-apps', '[]');

            if (!empty($userTopMenuApps)) {
                $topMenuApps = $userTopMenuApps;
            }
        }

        return [
            'vars' => [
                'background-color' => $backgroundColor,
                'background-color-to' => $backgroundColorTo,
                'current-app-background-color' => $this->config->getAppValue('current-app-background-color', '#444444'),
                'loader-color' => $this->config->getAppValue('loader-color', '#0e75ac'),
                'text-color' => $this->config->getAppValue('text-color', '#FFFFFF'),
                'opener' => $this->config->getAppValue('opener', 'side-menu-opener'),
                'icon-invert-filter' => abs($this->config->getAppValueInt('icon-invert-filter', '0')).'%',
                'icon-opacity' => abs($this->config->getAppValueInt('icon-opacity', '100') / 100),
            ],
            'display-logo' => $this->config->getAppValueBool('display-logo', 1),
            'opener-only' => $this->config->getAppValueBool('opener-only', 0),
            'external-sites-in-top-menu' => $this->config->getAppValueBool('external-sites-in-top-menu', 0),
            'size-icon' => $this->config->getAppValue('size-icon', 'normal'),
            'size-text' => $this->config->getAppValue('size-text', 'normal'),
            'always-displayed' => $this->config->getAppValueBool('always-displayed', '0'),
            'big-menu' => $this->config->getAppValueBool('big-menu', '0'),
            'top-menu-apps' => $topMenuApps,
        ];
    }
}
